using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Web.Mvc;
using HIPS.Web.Components.Collections;
using HIPS.Web.Model.AssistedRegistration;
using HIPS.Web.Model.Common;
using HIPS.Web.UI.Conversion.AssistedRegistration;

namespace HIPS.Web.UI.ViewModels.AssistedRegistration
{
    /// <summary>
    ///     View Model used for the PCEHR Register view.
    /// </summary>
    /// //TODO: Refactor to RegisterDependant + Register ViewModels
    public class RegisterViewModel
    {
        #region Model Data Properties

        [Required]
        [Display(Name = "Hospital")]
        public string HospitalCode { get; set; }

        /// <summary>
        ///     The registering Applicant.
        /// </summary>
        public PersonDemographicViewModel Applicant { get; set; }

        /// <summary>
        ///     The parent/guardian Representative for an Applicant who is a minor.
        ///     Required for Applicants under the age of 14, optional for Applicants 14-18.
        /// </summary>
        public PersonDemographicViewModel Representative { get; set; }

        [Required]
        [Display(Name = "Identity Verification Method")]
        public string IdentityVerificationMethod { get; set; }

        [Display(Name = "Medicare Access Consent")]
        public MedicareConsentViewModel MedicareConsents { get; set; }
       
        /// <summary>
        /// Applicant's Patient Master ID (used for retry submissions)
        /// </summary>
        public int? PatientMasterId { get; set; } 

        [Display(Name = "Consent")]
        public bool ConsentDeclared { get; set; }

        public string Phone { get; set; }
        public string Email { get; set; }
        public string IvcDeliveryMethod { get; set; }

        

        #endregion

        #region Data Source Properties

        public IEnumerable<SelectListItem> IdentityVerificationMethods { get; internal set; }
        public IEnumerable<SelectListItem> Hospitals { get; internal set; }

        public List<KeyValuePair<SelectListItem, List<SelectListItem>>> MedicareConsentOptions
        {
            get { return MedicareConsents.MedicareConsentOptions; }
            internal set { MedicareConsents.MedicareConsentOptions = value; }
        }

        #endregion

        #region Data Source Methods

        //TODO: General Design, these viewmodels shouldn't have model references. Refactor like HPII

        public IdentityVerificationMethod GetIdentityVerificationMethod(List<IdentityVerificationMethod> identityVerificationMethods)
        {
            return identityVerificationMethods.FirstOrDefault(i => i.Code == IdentityVerificationMethod);
        }

        public IvcDeliveryMethodDecision GetIvcDeliveryMethodDecision(List<IvcDeliveryMethod> ivcDeliveryMethods)
        {
            IvcDeliveryMethod ivcMethod = ivcDeliveryMethods.FirstOrDefault(i => i.Code == IvcDeliveryMethod);

            if (ivcMethod == null)
            {
                return null;
            }

            //TODO - Better switch/case, confirm support, regex, not more than one email+phone, support new methods with destinations
            if (ivcMethod.Code == "email")
            {
                return new IvcDeliveryMethodDecision(ivcMethod, Email, Email);
            }
            else if (ivcMethod.Code == "sms")
            {
                return new IvcDeliveryMethodDecision(ivcMethod, Phone, Phone);
            }
            else
            {
                return new IvcDeliveryMethodDecision(ivcMethod);
            }
        }

        public Hospital GetHospital(List<Hospital> hospitals)
        {
            return hospitals.FirstOrDefault(h => h.HospitalFacilityCode == HospitalCode);
        }

        /// <summary>
        ///     Populates the data source of Indigenous Statuses for this view model and nested entities.
        /// </summary>
        /// <param name="indigenousStatuses">A data source of possible Indigenous Statuses.</param>
        public void SetIndigenousStatuses(List<IndigenousStatus> indigenousStatuses)
        {
            Applicant.SetIndigenousStatuses(indigenousStatuses);
            Representative.SetIndigenousStatuses(indigenousStatuses);
        }

        /// <summary>
        ///     Populates the data source of Sexes for this view model and nested entities.
        /// </summary>
        /// <param name="sexes">A data source of possible Sexes.</param>
        public void SetSexes(List<Sex> sexes)
        {
            Applicant.SetSexes(sexes);
            Representative.SetSexes(sexes);
        }

        #endregion

        #region Constructors

        public RegisterViewModel()
        {
            Applicant = new PersonDemographicViewModel();
            Representative = new PersonDemographicViewModel { ShowDvaFileNumber = false, ShowIndigenousStatus = false };
            MedicareConsents = new MedicareConsentViewModel();
            MedicareConsentOptions = new List<KeyValuePair<SelectListItem, List<SelectListItem>>>();
        }

        public RegisterViewModel(List<Sex> sexes, List<IndigenousStatus> indigenousStatuses, List<IdentityVerificationMethod> identityVerificationMethods, List<Hospital> hospitals, List<MedicareConsent> medicareConsentOptions)
            : this(null, sexes, indigenousStatuses, identityVerificationMethods, hospitals, medicareConsentOptions)
        {
        }

        /// <summary>
        /// </summary>
        /// <param name="patient"></param>
        /// <param name="sexes"></param>
        /// <param name="indigenousStatuses"></param>
        /// <param name="identityVerificationMethods"></param>
        public RegisterViewModel(ValidatedPatient patient, List<Sex> sexes, List<IndigenousStatus> indigenousStatuses, List<IdentityVerificationMethod> identityVerificationMethods, List<Hospital> hospitals, List<MedicareConsent> medicareConsentOptions)
            : this()
        {
            SetSexes(sexes);
            SetIndigenousStatuses(indigenousStatuses);
            IdentityVerificationMethods = identityVerificationMethods.ToSelectListItems(i => i.Code, i => i.Description);
            Hospitals = hospitals.ToSelectListItems(i => i.HospitalFacilityCode, i => i.HospitalName);
            MedicareConsents = new MedicareConsentViewModel();
            MedicareConsentOptions = medicareConsentOptions.ToNestedSelectList();

            if (patient != null)
            {
                Applicant.LoadPatient(patient);
                HospitalCode = patient.CurrentLocation.Hospital.HospitalFacilityCode;
            }
        }

        #endregion
    }
}